import arcpy

class Toolbox(object):
	def __init__(self):
		"""Define the toolbox (the name of the toolbox is the name of the .pyt file)."""
		self.label = "Flow Map Toolbox"
		self.alias = "APL"
		self.tools = [FlowMapGenerator]

class FlowMapGenerator(object):
	def __init__(self):
		"""Define the tool (tool name is the name of the class)."""
		self.label = "FlowMapGenerator"
		self.description = "Generates a distributive flow map from one source to many destination points. If you input a polygon feature class for your source or destinations, centroids will be used as points. Make sure that your source, destination, and (optional) impassable feature classes are all projected."
		self.canRunInBackground = True

	def getParameterInfo(self):
		"""Define parameter definitions."""
		sourceParameter = arcpy.Parameter(
			displayName = "Source Point/Polygon Feature Class",
			name = "source",
			datatype = "Feature Layer",
			parameterType = "Required",
			direction = "Input")
		destinationParameter = arcpy.Parameter(
			displayName = "Destination Point/Polygon Feature Class",
			name = "destination",
			datatype = "Feature Layer",
			parameterType = "Required",
			direction = "Input")
		zValueParameter = arcpy.Parameter(
			displayName = "Z Value",
			name = "z_value",
			datatype = "Field",
			parameterType = "Required",
			direction = "Input")
		zValueParameter.parameterDependencies = [destinationParameter.name]
		impassableParameter = arcpy.Parameter(
			displayName = "Impassable Polygon Feature Class",
			name = "impassable",
			datatype = "Feature Layer",
			parameterType = "Optional",
			direction = "Input")
		extentParameter = arcpy.Parameter(
			displayName = "Processing Extent",
			name = "processing_extent",
			datatype = "Extent",
			parameterType = "Required",
			direction = "Input")
		cellSizeParameter = arcpy.Parameter(
			displayName = "Cell Size",
			name = "cell_size",
			datatype = "Cell Size",
			parameterType = "Required",
			direction = "Input")
		sourceWeightParameter = arcpy.Parameter(
			displayName = "Source Weight",
			name = "source_weight",
			datatype = "Long",
			parameterType = "Required",
			direction = "Input")
		sourceWeightParameter.value = 100
		destinationWeightParameter = arcpy.Parameter(
			displayName = "Destination Weight",
			name = "destination_weight",
			datatype = "Long",
			parameterType = "Required",
			direction = "Input")
		destinationWeightParameter.value = 100
		gridWeightParameter = arcpy.Parameter(
			displayName = "Grid Weight",
			name = "grid_weight",
			datatype = "Long",
			parameterType = "Required",
			direction = "Input")
		gridWeightParameter.value = 100
		gridSizeParameter = arcpy.Parameter(
			displayName = "Grid Size",
			name = "grid_size",
			datatype = "Long",
			parameterType = "Required",
			direction = "Input")
		gridSizeParameter.value = 5
		flowMapParameter = arcpy.Parameter(
			displayName = "Output Line Feature Class",
			name = "flow_map",
			datatype = "Feature Class",
			parameterType = "Required",
			direction = "Output")
		params = [sourceParameter, destinationParameter, zValueParameter, impassableParameter, extentParameter, cellSizeParameter, sourceWeightParameter, destinationWeightParameter, gridWeightParameter, gridSizeParameter, flowMapParameter]
		return params

	def isLicensed(self):
		"""Set whether tool is licensed to execute."""
		return True

	def updateParameters(self, parameters):
		"""Modify the values and properties of parameters before internal validation is performed. This method is called whenever a parameter has been changed."""
		return

	def updateMessages(self, parameters):
		"""Modify the messages created by internal validation for each tool parameter. This method is called after internal validation."""
		return

	def execute(self, parameters, messages):
		"""The source code of the tool."""

		# Assign variables for parameters.
		sourceFeature = parameters[0]
		destinationFeature = parameters[1]
		zValue = parameters[2]
		impassable = parameters[3]
		extent = parameters[4]
		cellSize = parameters[5]
		sourceWeight = parameters[6]
		destinationWeight = parameters[7]
		gridWeight = parameters[8]
		gridSize = parameters[9]

		# Split extent into usable values.
		leftExtent = str(extent.value).split()[0]
		bottomExtent = str(extent.value).split()[1]
		rightExtent = str(extent.value).split()[2]
		topExtent = str(extent.value).split()[3]

		# Environmental settings for raster calculations later.
		arcpy.env.cellSize = cellSize.value
		arcpy.env.extent = extent.value

		# Describe source feature class.
		desc = arcpy.Describe(sourceFeature.value)

		# Get the shape type ("Polygon", "Polyline", etc.) of the feature class.
		type = desc.shapeType

		# If "Polygon", run Feature to Point.
		if type == "Polygon":
			arcpy.FeatureToPoint_management(sourceFeature.value, "in_memory\sourcePoint", "CENTROID")
		else:
			arcpy.CopyFeatures_management(sourceFeature.value, "in_memory\sourcePoint")
		
		# Describe destination feature class.
		desc = arcpy.Describe(destinationFeature.value)

		# Get the shape type ("Polygon", "Polyline", etc.) of the feature class
		type = desc.shapeType

		# If "Polygon", run Feature to Point.
		if type == "Polygon":
			arcpy.FeatureToPoint_management(destinationFeature.value, "in_memory\destinationPoints", "CENTROID")
		else:
			arcpy.CopyFeatures_management(destinationFeature.value, "in_memory\destinationPoints")

		# Remove destinations that have Z-value of NULL or 0 using UpdateCursor.
		with arcpy.da.UpdateCursor("in_memory\destinationPoints", str(zValue.value)) as cursor:
			for row in cursor:
				if (row[0] is None or row[0] == 0):
					cursor.deleteRow()

		# Check that there's only one source, return error if more than one.
		sourceCount = int(arcpy.GetCount_management("in_memory\sourcePoint").getOutput(0))

		# Return error if more than one source.
		if sourceCount > 1:
			arcpy.AddError("Source feature class contains more than one feature.")

		# Check that there's more than one destination.
		destinationCount = int(arcpy.GetCount_management("in_memory\destinationPoints").getOutput(0))

		# Return error if only one destination.
		if destinationCount == 1:
			arcpy.AddError("Destination feature class only contains one feature.")

		# Create variable for grid origin, Y-axis, opposite corner.
		gridOrigin = str(leftExtent + " " + bottomExtent)
		gridYAxis = str(leftExtent + " " + topExtent)
		gridOpposite = str(rightExtent + " " + topExtent)

		# Create variable for grid size.
		gridSize = int(str(cellSize.value)) * gridSize.value

		# Create grid.
		arcpy.CreateFishnet_management("in_memory\grid", gridOrigin, gridYAxis, gridSize, gridSize, 0, 0, gridOpposite, "NO_LABELS", "", "POLYGON")

		# Convert grid to points.
		arcpy.FeatureToPoint_management("in_memory\grid", "in_memory\gridPoints", "CENTROID")

		# Get euclidean distance from grid points.
		gridDistanceRaster = arcpy.sa.EucDistance("in_memory\gridPoints")

		# Slice into 100 sections.
		gridSlicedRaster = arcpy.sa.Slice(gridDistanceRaster, gridWeight.value)

		# Get euclidean distance from source points.
		sourceDistanceRaster = arcpy.sa.EucDistance("in_memory\sourcePoint")

		# Slice into 100 sections.
		sourceSlicedRaster = arcpy.sa.Slice(sourceDistanceRaster, sourceWeight.value)

		# Get euclidean distance from destination points.
		destinationDistanceRaster = arcpy.sa.EucDistance("in_memory\destinationPoints")

		# Slice into 100 sections.
		destinationSlicedRaster = arcpy.sa.Slice(destinationDistanceRaster, destinationWeight.value)

		# Sum sliced surfaces.
		sumSlicedRaster = gridSlicedRaster + sourceSlicedRaster + destinationSlicedRaster

		# If there's an impassable feature, convert it to raster, set cell values to two billion, and add to sumSlicedRaster.
		if impassable.value is not None:

			# Describe impassable feature class.
			desc = arcpy.Describe(sourceFeature.value)

			# Get the shape type ("Polygon", "Polyline", etc.) of the feature class
			type = desc.shapeType

			# Convert to raster.
			if type == "Polygon":
				arcpy.PolygonToRaster_conversion(impassable.value, "OBJECTID", "in_memory\impassable")
			if type == "Point":
				arcpy.PointToRaster_conversion(impassable.value, "OBJECTID", "in_memory\impassable")
			if type == "Polyline":
				arcpy.PolylineToRaster_conversion(impassable.value, "OBJECTID", "in_memory\impassable")

			# Set impassable areas to cost of two billion, add to sumSlicedRaster.
			impassableRaster = arcpy.sa.Con("in_memory\impassable", 2000000000)

			sumSlicedRaster = arcpy.sa.Con(arcpy.sa.IsNull(impassableRaster), sumSlicedRaster, impassableRaster)

		# Get cost back link.
		costBackLink = arcpy.sa.CostBackLink("in_memory\sourcePoint", sumSlicedRaster)

		# Get cost path.
		costPath = arcpy.sa.CostPath("in_memory\destinationPoints", sumSlicedRaster, costBackLink)

		# Create path distance raster.
		pathDistance = arcpy.sa.PathDistance ("in_memory\sourcePoint", sumSlicedRaster)

		# Substitute cost path values for path distance values.
		pathDistancePath = arcpy.sa.Con(costPath, pathDistance)

		# Get highest value from path distance raster.
		maximumPathDistanceValue = arcpy.GetRasterProperties_management(pathDistance, "MAXIMUM")

		# Multiply that highest value by ten, store that value.
		impassableValue = float(maximumPathDistanceValue.getOutput(0)) * 2

		# Create cost surface - set null values on pathDistancePath to impassableValue, non-null values stay the same.
		costSurface = arcpy.sa.Con(arcpy.sa.IsNull(pathDistancePath), impassableValue, pathDistancePath)

		# Run flow direction.
		flowDirection = arcpy.sa.FlowDirection(costSurface)

		# Run point to raster on destination points.
		arcpy.PointToRaster_conversion("in_memory\destinationPoints", str(zValue.value), "in_memory\pointToRaster", "SUM")

		# Set NULL values to zero.
		costSurface = arcpy.sa.Con(arcpy.sa.IsNull("in_memory\pointToRaster"), 0, "in_memory\PointToRaster")

		# Run flow accumulation.
		flowAccumulation = arcpy.sa.FlowAccumulation(flowDirection, costSurface)

		# Multiply accumulation values by 100 so we lose less data when we convert to integer.
		flowAccumulation = flowAccumulation * 100

		# Convert flowAccumulation to integer values.
		integerFlow = arcpy.sa.Int(flowAccumulation)

		# Convert all zero values to null.
		flowAccumulationNull = arcpy.sa.SetNull(integerFlow, integerFlow, "VALUE = 0")

		# Run stream to feature.
		arcpy.sa.StreamToFeature(flowAccumulationNull, flowDirection, "in_memory\streamToFeature", "NO_SIMPLIFY")

		# Create new float attribute in streamToFeature.
		arcpy.AddField_management("in_memory\streamToFeature", "Z_VALUE", "DOUBLE")

		# Divide "GRID_CODE" attribute by 100 to get true values back as "Z_VALUE".
		with arcpy.da.UpdateCursor("in_memory\streamToFeature", ("GRID_CODE", "Z_VALUE")) as cursor:
			for row in cursor:
				row[1] = row[0] / 100.0
				cursor.updateRow(row)

		# Delete "GRID_CODE".
		arcpy.DeleteField_management("in_memory\streamToFeature", "GRID_CODE")

		# Copy streamToFeature into output parameter of tool.
		arcpy.CopyFeatures_management("in_memory\streamToFeature", parameters[10].value)

		return








